define([
    'underscore',
    'backbone',
    'marionette',
    'modules/new-appointment-request/views/direct/clinics/layout-view',
    'modules/new-appointment-request/views/direct/preferred-date/preferred-date-layout',
    'modules/form/question-model',
    'modules/form/question-view',
    'text!modules/new-appointment-request/views/direct/template.html',
    'modules/new-appointment-request/resources/direct/custom-messages/model',
], function(
    _,
    Backbone,
    Mn,
    ClinicsView,
    DateTimeView,
    QuestionModel,
    QuestionView,
    template,
    CustomMessages
) {
    'use strict';


    var DataModel = Backbone.Model.extend({
        isReady: function() {
            return this.isAppointmentValid() &&
                this.isDateTimeValid();
        },
        isAppointmentValid: function() {
            return this.get('apptLength') > 0;
        },

        isDateTimeValid: function() {
            return !_.isEmpty(this.get('dateTime'));
        },
    });

    return Mn.View.extend({
        tagName: 'fieldset',

        className: 'appointment-details-section',

        template: _.template(template),

        regions: {
            clinics: '.appointment-details-clinics-section',
            purpose: '.appointment-details-purpose-section',
            dateTime: '.appointment-details-date-time-section',
            email: '.appointment-details-email',
        },

        modelEvents: {'change:clinic': '_onClinicSelect'},

        purposeEvents: {
            'change:value': 'updateFDNS   elPurpose',
            'change': 'validate',
        },

        dataEvents: {'change': 'validate'},

        childViewEvents: {'no:clinics:found': 'noClinicsFound'},

        noClinicsFound: function() {
            var facility = this.model.get('facility');
            facility.set('directSchedulingSupported', false);
            this.model.trigger('change:facility');


            // Stop trying to build the direct appointments section
            this.destroy();
        },

        resetStaticRegions: function() {
            if (this.data) {
                this.getRegion('dateTime').reset();
                this.data = new DataModel();
                this.bindEvents(this.data, this.dataEvents);
            }
        },

        initialize: function() {
            this.purposeModel = this.createPurposeModel();
            this.data = new DataModel();
            this.customMessage = new CustomMessages();
            this.customMessageXhr = this.customMessage.fetchAppointmentsMessage(this.model.getInstitutionCode());
            this.bindEvents(this.purposeModel, this.purposeEvents);
            this.bindEvents(this.data, this.dataEvents);
        },

        onRender: function() {
            var region = this.getRegion('clinics');
            var clinics = this.options.clinicsCollection;
            var options = {
                model: this.model,
                customMessages: new CustomMessages(),
            };

            if (clinics) {
                options.clinicsCollection = clinics;
            }
            this.model.unset('clinic', {silent: true});
            region.show(new ClinicsView(options));
        },

        onBeforeDestroy: function() {
            this.unbindEvents(this.purposeModel, this.purposeEvents);
            this.unbindEvents(this.pdata, this.dataEvents);
        },

        updateFDNS   elPurpose: function(model) {
            this.model.set('purpose', model.get('value'));
        },

        /**
         * Factory function for creating a purpose QuestionModel for the QuestionView
         * @param {*} [attributes] Additional data or Default Override data
         * @param {*} [options]
         * @return {Backbone.Model}
         */
        createPurposeModel: function(attributes, options) {
            var defaults = {
                id: 'purpose',
                class: 'form-native-controlgroup form-native-controlgroup-vertical',
                type: 'textarea',
                label: 'Reason for Appointment:',
                maxlength: 150,
                required: true,
            };
            _.extend(defaults, attributes);

            return new QuestionModel(defaults, options || {});
        },

        /**
         * Called from a Backbone Model change:clinic event.
         * @param {Backbone.Model} model
         * @param {*} clinic
         * @param {Object} options
         * @return {void}
         * @private
         */
        _onClinicSelect: function(model, clinic, options) {
            var clinicId;

            if (this.model.get('clinic')) {
                clinicId = this.model.getClinicId();
                this._showPurposeSectionView(model, clinicId, options);
                this._afterMessageShowDateTime(model, clinicId, options);
                return;
            }
            this.resetStaticRegions();
        },

        _afterMessageShowDateTime: function(model, clinicId) {
            var showWithArgs = _.partial(this._showDateTimeLayoutView, model, clinicId);
            var show = _.bind(showWithArgs, this);

            $.when(this.customMessageXhr).done(show);
        },

        /**
         * Called from a Backbone Model change:clinic event
         * @param {Backbone.Model} model
         * @param {string} clinicId
         * @return {void}
         * @private
         */
        _showDateTimeLayoutView: function(model, clinicId) {
            var clinicName = this.model.getClinicName();
            var clinicsRegion = this.getRegion('clinics');
            var dateTimeRegion = this.getRegion('dateTime');

            var view = clinicsRegion.currentView;
            var slotsModel = view.getTimeSlots(clinicId);
            var slots = slotsModel.get('appointmentTimeSlot');

            this.model.unset('dateTime', {silent: true});
            this.model.unset('desiredDate', {silent: true});

            if (!this.model.isExpressCare()) {
                dateTimeRegion.show(new DateTimeView({
                    model: this.data,
                    collection: slots,
                    customMessage: this.customMessage,
                }));
            }

            if (typeof gas !== 'undefined') {
                gas('send', 'event', 'veteran-appointment', 'clinic-selected', clinicName);
            }

        },

        /**
         * Called from a Backbone Model change:clinic event
         * @return {void}
         * @private
         */
        _showPurposeSectionView: function() {
            var region = this.getRegion('purpose');

            this.model.unset('purpose', {silent: true});
            region.show(new QuestionView({model: this.purposeModel}));
        },

        validate: function() {
            var purpose = this.purposeModel.get('value');
            var data;

            if (!_.isEmpty(purpose) && this.data.isReady()) {
                data = this.data.toJSON();
                data.purpose = purpose;
                this.model.set('direct', data);
                return true;
            }
            this.model.unset('direct');
            return false;
        },
    });
});
